library(shiny)
library(tidyverse)

# Load Dataset Function (modified to handle numeric Likert scale)
load_dataset <- function(file_path) {
  read_csv(file_path,
           col_types = cols(
             tweet_id = "c",
             media_name = "c",
             AnnotatorChecked = col_logical(),
             AI_generated_verification = col_logical(),
             AI_generated_confidence = col_double(),
             believable = col_double(), # Change believable to a numeric type (Likert scale)
             harmful = col_double(),
             sentiment = col_double(),
             technology = col_logical(),
             health = col_logical(),
             politics = col_logical(),
             crime = col_logical(),
             business = col_logical(),
             entertainment = col_logical(),
             sports = col_logical(),
             education = col_logical(),
             satire = col_logical(),
             other = col_logical(),
             PresentationCorrect = col_logical()
             # Change harmful to a numeric type (Likert scale)
           ), locale = locale(encoding = "UTF-8")
  ) |>
    dplyr::mutate(
      tweet_id = bit64::as.integer64(tweet_id), #|>
      #    media_type == "photo" ~ paste0(media_name, ".jpg"),  # Assuming images are .jpg
      #    media_type == "video" ~ paste0(media_name, ".mp4"),  # Assuming videos are .mp4
      #    TRUE ~ media_name  # Keep as is if no media_type
      #  ),
      MediaPath = paste0("data/media/", media_name),
      MediaExists = file.exists(MediaPath)
    )
}

# Save Dataset Function (same as before)
save_dataset <- function(data, file_path) {
  write_csv(data, file_path)
}

# Initialize Reactive Values Function (same as before)
initialize_reactive_values <- function(data) {
  reactiveValues(
    videoNum = 1
  )
}

# Dataset Path (same as before)
dataset_path_in <- fs::path_abs(
  here::here("Shiny_Deepfakes", "data", "tweets.csv")
)

data <- load_dataset(dataset_path_in)

# Define Server
shinyServer(function(input, output, session) {

  # Output path
  epoch_now <- 
    lubridate::now(tzone="UTC") |> 
    lubridate::floor_date("second") |> 
    (`-`)(lubridate::ymd_hms("1970-01-01T00:00:00", tz="UTC")) |> 
    lubridate::time_length(unit="second") |> 
    as.character()

  dataset_path_out <- fs::path_abs(
    here::here("Shiny_Deepfakes", "app_data", epoch_now, "tweets.csv")
  )
  
  fs::dir_create(fs::path_dir(dataset_path_out))

  # Reactive values for current video/tweet
  v <- initialize_reactive_values(data)
  
  # Fetch the current data (video or tweet)
  current_data <- reactive({
    req(data, v$videoNum)
    validate(
      need(v$videoNum <= nrow(data) && v$videoNum > 0, "Invalid video number!")
    )
    slice(data, v$videoNum)
  })
  
  # Progress bar
  output$progressCounter <- renderUI({
    req(data, v$videoNum)
    total <- nrow(data)
    div(
      style = "text-align: center; font-size: 18px; font-weight: bold; margin-bottom: 10px;",
      paste("Progress:", v$videoNum, "/", total)
    )
  })
  
  # Output the UI for the current tweet/video
  output$tweetDisplay <- renderUI({
    req(current_data())
    
    media_filename <- basename(current_data()$media_name)
    media_path <- file.path("data/media", media_filename) 
    www_media_path <- file.path("www/media", media_filename)
    
    if (file.exists(media_path) && !file.exists(www_media_path)) {
      dir.create("www/media", showWarnings = FALSE, recursive = TRUE)
      file.copy(media_path, www_media_path, overwrite = TRUE)
    }
    
    media_url <- paste0("media/", media_filename)
    
    # UI Output
    tagList(
      div(
        class = "tweet-box",
        div(
          style = "display: flex; justify-content: center; gap: 10px; align-items: center;",
          img(
            src = "https://abs.twimg.com/sticky/default_profile_images/default_profile_normal.png",
            width = "40px", height = "40px", style = "border-radius: 50%;"
          ),
          div(
            strong(current_data()$username),
            br(),
            span(style = "color: gray;", current_data()$handle, current_data()$timestamp)
          )
        ),
        br(),
        
        # Tweet/Video text
        div(class = "tweet-text", current_data()$text),
        br(),
        
        # Display media (if exists and is of valid type)
        if (!is.null(media_filename)) {
          if (grepl("\\.mp4$", media_filename, ignore.case = TRUE)) {
            div(
              style = "display: block; width: 600px; margin: 0 auto;",
              tags$video(
                controls = TRUE, autoplay = FALSE, width = "100%", height = "auto",
                style = "max-width: 100%; border-radius: 10px;",
                tags$source(src = media_url, type = "video/mp4"),
                "Your browser does not support the video tag."
              )
            )
          } else if (grepl("\\.jpg$|\\.png$", media_filename, ignore.case = TRUE)) {
            img(
              src = media_url,
              width = "80%",
              style = "display: block; margin: 0 auto; max-width: 100%; border-radius: 10px;"
            )
          }
        } else {
          p("No media available for this post.")
        }
      )
    )
  })
  
  
  # Output the annotation form (Likert scale for Believability, Harmfulness, Sentiment, and Topics)
  # Render Left Column of Annotation Form (Checkboxes)
  output$annotationFormLeft <- renderUI({
    req(current_data())
    
    tagList(
      checkboxGroupInput("PresentationCorrect",
                         "Is the media attached to the post presented correctly?",
                         choices = list("Yes" = "yes", "No" = "no"),
                         selected = NULL
      ),
      
      checkboxGroupInput("AI_generated_verification",
                         "To the best of your knowledge, do you think the media presented in the above post was generated using AI?",
                         choices = list("Yes" = "yes", "No" = "no"),
                         selected = NULL
      ),
      
      div(
        style = "text-align: center;",
        sliderInput("AI_generated_confidence",
                    "How confident are you regarding your assessment of the media's origin (AI-generated or not)?",
                    min = 1, max = 5, value = 3, step = 1, animate = TRUE, ticks = FALSE, round = TRUE
        ),
        div(
          style = "display: flex; justify-content: space-between; width: 100%;",
          tags$span("Not at all confident"), 
          tags$span("Extremely confident")
        )
      ),
      
      checkboxGroupInput("topics",
                         "What topics are covered in this post? (Select all that apply)",
                         choices = list(
                           "Technology" = "technology",
                           "Health" = "health",
                           "Politics" = "politics",
                           "Crime" = "crime",
                           "Business" = "business",
                           "Entertainment" = "entertainment",
                           "Sports" = "sports",
                           "Education" = "education",
                           "Satire" = "satire",
                           "Other" = "other"
                         )
      )
    )
  })
  
  # Render Right Column of Annotation Form (Sliders)
  output$annotationFormRight <- renderUI({
    req(current_data())
    tweet_data <- current_data()
    
    tagList(
      div(
        style = "text-align: center;",
        sliderInput("sentiment",
                    "What is the sentiment of this post?",
                    min = 1, max = 5, value = ifelse(is.na(tweet_data$sentiment), 3, tweet_data$sentiment),
                    step = 1, animate = TRUE, ticks = FALSE, round = TRUE
        ),
        div(
          style = "display: flex; justify-content: space-between; width: 100%;",
          tags$span("Very Negative"), 
          tags$span("Very Positive")
        )
      ),
      
      div(
        style = "text-align: center;",
        sliderInput("believable",
                    "If this post was widely spread, would its message likely be believed by many?",
                    min = 1, max = 5, value = ifelse(is.na(tweet_data$believable), 3, tweet_data$believable),
                    step = 1, animate = TRUE, ticks = FALSE, round = TRUE
        ),
        div(
          style = "display: flex; justify-content: space-between; width: 100%;",
          tags$span("Not at all believable"), 
          tags$span("Extremely believable")
        )
      ),
      
      div(
        style = "text-align: center;",
        sliderInput("harmful",
                    "If many believed this post, would it cause considerable harm?",
                    min = 1, max = 5, value = ifelse(is.na(tweet_data$harmful), 3, tweet_data$harmful),
                    step = 1, animate = TRUE, ticks = FALSE, round = TRUE
        ),
        div(
          style = "display: flex; justify-content: space-between; width: 100%;",
          tags$span("Not at all harmful"), 
          tags$span("Extremely harmful")
        )
      )
    )
  })
  # Save the annotations for the current video/tweet
  save_annotations <- function() {
    req(current_data())
    tweet_data <- current_data()
    
    # Update the dataset with the annotations
    data[v$videoNum, "AI_generated_verification"] <<- ifelse("yes" %in% input$AI_generated_verification, TRUE, FALSE)
    data[v$videoNum, "AI_generated_confidence"] <<- input$AI_generated_confidence
    data[v$videoNum, "sentiment"] <<- input$sentiment
    data[v$videoNum, "believable"] <<- input$believable
    data[v$videoNum, "harmful"] <<- input$harmful
    data[v$videoNum, "AnnotatorChecked"] <<- TRUE
    data[v$videoNum, "PresentationCorrect"] <<- ifelse("yes" %in% input$PresentationCorrect, TRUE, FALSE)
    
    # Update topic selections
    topic_cols <- c(
      "technology", "health", "politics", "crime", "business",
      "entertainment", "sports", "education", "satire", "other"
    )
    
    for (topic in topic_cols) {
      data[v$videoNum, topic] <<- topic %in% input$topics
    }
  }
  
  # Navigate to the previous video/tweet
  observeEvent(input$btn_prev, {
    v$videoNum <- max(1, v$videoNum - 1)
  })
  
  # Navigate to the next video/tweet and save annotations
  observeEvent(input$btn_next, {
    save_annotations()
    v$videoNum <- min(nrow(data), v$videoNum + 1)
  })
  
  # Finish the annotations and save the data
  observeEvent(input$btn_done, {
    save_annotations()
    save_dataset(data, dataset_path_out)
    showModal(modalDialog(
      title = "Thank You!",
      "Your responses have been saved.",
      easyClose = TRUE
    ))
  })
})

# Run the application
# shinyApp(ui, server)
